library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.ALL;

entity dummy is
    generic (
        C_DATA_WIDTH : integer := 32
    );
    port (
        aclk            : in std_logic;
        aresetn         : in std_logic;
        
        enable          : in std_logic;
        N_in            : in std_logic_vector(31 downto 0);
        N_out           : in std_logic_vector(31 downto 0);

        s_axis_tdata    : in  std_logic_vector(C_DATA_WIDTH-1 downto 0);
        s_axis_tvalid   : in  std_logic;
        s_axis_tlast    : in  std_logic;
        s_axis_tready   : out std_logic;

        m_axis_tdata    : out std_logic_vector(C_DATA_WIDTH-1 downto 0);
        m_axis_tvalid   : out std_logic;
        m_axis_tlast    : out std_logic;
        m_axis_tready   : in  std_logic
    );
end dummy;

architecture behv of dummy is
    component fifo_generator_0 IS
      Port (
        wr_rst_busy     : OUT STD_LOGIC;
        rd_rst_busy     : OUT STD_LOGIC;

        s_aclk          : IN STD_LOGIC;
        s_aresetn       : IN STD_LOGIC;

        s_axis_tvalid   : IN STD_LOGIC;
        s_axis_tready   : OUT STD_LOGIC;
        s_axis_tlast    : IN STD_LOGIC;
        s_axis_tdata    : IN STD_LOGIC_VECTOR(31 DOWNTO 0);

        m_axis_tvalid   : OUT STD_LOGIC;
        m_axis_tready   : IN STD_LOGIC;
        m_axis_tlast    : OUT STD_LOGIC;
        m_axis_tdata    : OUT STD_LOGIC_VECTOR(31 DOWNTO 0)
      );
    end component;

    signal s_valid      : std_logic;
    signal s_ready      : std_logic;
    signal s_data       : std_logic_vector(C_DATA_WIDTH-1 downto 0) := (others => '0');

    signal m_valid      : std_logic;
    signal m_ready      : std_logic;
    signal m_last       : std_logic;
    signal m_data       : std_logic_vector(C_DATA_WIDTH-1 downto 0) := (others => '0');

    signal write_count  : std_logic_vector(31 downto 0) := (others => '0');
    
    signal system_reset : std_logic := '0';
begin
    
    system_reset <= not( (not aresetn) or (not enable) );
   
    data_fifo_in : fifo_generator_0
    port map (
        wr_rst_busy     => open,
        rd_rst_busy     => open,

        s_aclk          => aclk,
        s_aresetn       => system_reset,

        s_axis_tvalid   => s_axis_tvalid,
        s_axis_tready   => s_axis_tready,
        s_axis_tlast    => s_axis_tlast,
        s_axis_tdata    => s_axis_tdata,

        m_axis_tvalid   => s_valid,
        m_axis_tready   => s_ready,
        m_axis_tlast    => open,
        m_axis_tdata    => s_data
    );

    process(aclk, aresetn)
        variable write_count : std_logic_vector(31 downto 0);
    begin
        if system_reset = '0' then
            write_count := (others => '0');
            m_valid <= '0';
            m_last  <= '0';
            s_ready <= '0';
        elsif rising_edge(aclk) then
            if (unsigned(write_count) < unsigned(N_out)) and (s_valid = '1') and (m_ready = '1') then
                m_valid <= '1';
                s_ready <= '1';
                write_count := std_logic_vector(unsigned(write_count) + 1);
                
                if unsigned(write_count) = unsigned(N_out) - 1 then
                    m_last <= '1';
                else
                    m_last <= '0';
                end if;
                
            elsif unsigned(write_count) = unsigned(N_out) then
                write_count := (others => '0');
                m_valid <= '0';
                s_ready <= '0';
            end if;
        end if;
    end process;

    m_data <= s_data;

    data_fifo_out : fifo_generator_0
    port map (
        wr_rst_busy     => open,
        rd_rst_busy     => open,

        s_aclk          => aclk,
        s_aresetn       => system_reset,

        s_axis_tvalid   => m_valid,
        s_axis_tready   => m_ready,
        s_axis_tlast    => m_last,
        s_axis_tdata    => m_data,

        m_axis_tvalid   => m_axis_tvalid,
        m_axis_tready   => m_axis_tready,
        m_axis_tlast    => m_axis_tlast,
        m_axis_tdata    => m_axis_tdata
    );
end behv;
