#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/mman.h>

#include "dummy.h"
#include "libpdpu.h"    // used for the status flags

/*
 * This is the TX thread. It reads a given amount of data from an input file
 * stored in SD and issues a DMA transfer request from PS to PL through
 * the AXI DMA IP Core.
 * @args: The list of arguments to pass.
 */
void *ps2pl(void *args) {
    int buf_id = 0;

    dma_thread_args *thread_args = (dma_thread_args *)args;
    Channel channel = *(thread_args->channel);

    uint32_t n_read = 0;
    uint32_t transfer = thread_args->transfer_size;
    uint32_t total = thread_args->total_size;
    uint32_t n, remaining_bytes;

    int fi;

    if ((fi = open(thread_args->file, O_RDONLY)) == -1) {
        thread_args->status = ERR;
        return NULL;
    }

    while (*(thread_args->halt_op) == 0) {
        if (transfer > total) {
            transfer = total;
        }

        // Read data from the input file and place them to the DMA buffer
        n = read(fi, channel.buf_ptr[buf_id].buffer, transfer);

        if (n == 0) {
            break; // No more bytes available to read, exit the loop
        }

        channel.buf_ptr[buf_id].length = n; // Set the transfer size

        n_read += n; // Update the read bytes counter

        ioctl(channel.fd, START_XFER, &buf_id);   // Initiate a transfer
        ioctl(channel.fd, FINISH_XFER, &buf_id);  // Block until completion

        if (channel.buf_ptr[buf_id].status != PROXY_NO_ERROR) {
            // If abnormal DMA termination occurred exit
            thread_args->status = ERR;
            *(thread_args->halt_op) = 1;
            break;
        }

        if (n_read == total) {
            break; // If all payload bytes are transferred, break the loop
        }

        remaining_bytes = total - n_read;

        if (transfer > remaining_bytes) {
            transfer = remaining_bytes;
        }
    }

    close(fi);
    thread_args->status = OK;

    return NULL;
}

/*
 * This is the RX thread. It reads from a DMA peripheral device and writes the
 * expected data to an output file in SD.
 * @args: The list of arguments to pass.
 */
void *pl2ps(void *args) {
    int buf_id = 0;

    dma_thread_args *thread_args = (dma_thread_args *)args;
    Channel channel = *(thread_args->channel);

    uint32_t n_write = 0;
    uint32_t transfer = thread_args->transfer_size;
    uint32_t total = thread_args->total_size;
    uint32_t remaining_bytes;

    int fo;

    if ((fo = open(thread_args->file, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR)) == -1) {
        thread_args->status = ERR;
        return NULL;
    }

    while (*(thread_args->halt_op) == 0) {
        if (transfer > total) {
            transfer = total;
        }

        channel.buf_ptr[buf_id].length = transfer; // Set the transfer size

        ioctl(channel.fd, START_XFER, &buf_id);     // Initiate a transfer
        ioctl(channel.fd, FINISH_XFER, &buf_id);    // Block until completion

        if (channel.buf_ptr[buf_id].status != PROXY_NO_ERROR) {
            // If abnormal DMA termination occurred, exit
            thread_args->status = ERR;
            *(thread_args->halt_op) = 1;
            break;
        }

        // Write data to output file
        n_write += write(fo, channel.buf_ptr[buf_id].buffer, transfer);

        if (n_write == total) {
            break; // If all payload bytes are transferred, break the loop
        }

        remaining_bytes = total - n_write;

        if (transfer > remaining_bytes) {
            transfer = remaining_bytes;
        }
    }

    close(fo);
    thread_args->status = OK;

    return NULL;
}
